<?php
require __DIR__ . '/vendor/autoload.php';

// Google API PHP Client (auth)
// Note: this is \Google_Client (no namespace), as provided by google/apiclient 2.0

/**
 * Generic helper to get an access token for given scopes using a service account.
 *
 * @param string $serviceAccountFile
 * @param array  $scopes
 * @return string
 * @throws Exception
 */
function getGoogleAccessToken($serviceAccountFile, array $scopes)
{
    /** @var \Google_Client $client */
    $client = new \Google_Client();
    $client->setAuthConfig($serviceAccountFile);
    $client->setScopes($scopes);

    $token = $client->fetchAccessTokenWithAssertion();

    if (isset($token['access_token'])) {
        return $token['access_token'];
    }

    throw new Exception('Unable to get access token: ' . json_encode($token));
}

/**
 * Get access token for FCM (firebase.messaging scope).
 */
function getFcmAccessToken($serviceAccountFile)
{
    return getGoogleAccessToken(
        $serviceAccountFile,
        ['https://www.googleapis.com/auth/firebase.messaging']
    );
}

/**
 * Get access token for Firestore (datastore scope).
 */
function getFirestoreAccessToken($serviceAccountFile)
{
    return getGoogleAccessToken(
        $serviceAccountFile,
        ['https://www.googleapis.com/auth/datastore']
    );
}

function sendFCMv1($title, $message, $topic)
{
    // ==== CONFIG ====
    $projectId = 'arqana-bd662'; // e.g. my-app-123456
    $serviceAccountFile = __DIR__ . '/arqana-bd662-firebase-adminsdk-fbsvc-d8399e5e41.json'; // path to JSON

    // Get access token
    $accessToken = getFcmAccessToken($serviceAccountFile);

    // v1 endpoint
    $url = "https://fcm.googleapis.com/v1/projects/$projectId/messages:send";

    // Build the message payload
    $body = [
        'message' => [
            'topic' => $topic,   // just the topic name, no /topics/
            'notification' => [
                'title' => $title,
                'body' => $message,
            ],
            // Optional Android config if you want high priority explicitly
            'android' => [
                'priority' => 'high',
                'notification' => [
                    'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                    'sound' => 'default',
                ],
            ],
            // Optional APNS config for iOS
            'apns' => [
                'payload' => [
                    'aps' => [
                        'sound' => 'default',
                        'content-available' => 1,
                    ],
                ],
            ],
        ],
    ];

    $jsonBody = json_encode($body);

    $headers = [
        'Authorization: Bearer ' . $accessToken,
        'Content-Type: application/json; charset=utf-8',
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonBody);

    $result = curl_exec($ch);

    if ($result === false) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('cURL error: ' . $error);
    }

    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    return [
        'http_code' => $httpCode,
        'response' => $result,
    ];
}
