<?php
/**
 * Create Scheduled Notification Endpoint
 *
 * Accepts POST requests to create a scheduled notification entry in Firestore.
 */

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json; charset=utf-8');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'error' => 'Method not allowed. Only POST requests are accepted.'
    ]);
    exit();
}

require __DIR__ . '/fcm_helper.php';

$projectId = 'arqana-bd662';
$serviceAccountFile = __DIR__ . '/arqana-bd662-firebase-adminsdk-fbsvc-d8399e5e41.json';

function csn_parseRequestBody()
{
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';

    if (strpos($contentType, 'application/json') !== false) {
        $rawBody = file_get_contents('php://input');
        $data = json_decode($rawBody, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return null;
        }
        return $data;
    }

    return $_POST;
}

function csn_validateParameters($data)
{
    $errors = [];

    if (empty($data['title'])) {
        $errors[] = 'title is required';
    }
    if (empty($data['message'])) {
        $errors[] = 'message is required';
    }
    if (empty($data['topic'])) {
        $errors[] = 'topic is required';
    }
    if (empty($data['scheduled_at'])) {
        $errors[] = 'scheduled_at is required';
    } else {
        try {
            // Try to parse as DateTime
            new DateTime($data['scheduled_at']);
        } catch (Exception $e) {
            $errors[] = 'scheduled_at must be a valid ISO-8601 date-time string (UTC preferred)';
        }
    }

    return $errors;
}

$requestData = csn_parseRequestBody();

if ($requestData === null) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Invalid JSON in request body.'
    ]);
    exit();
}

$validationErrors = csn_validateParameters($requestData);

if (!empty($validationErrors)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Validation failed: ' . implode(', ', $validationErrors)
    ]);
    exit();
}

$title = trim($requestData['title']);
$message = trim($requestData['message']);
$topic = trim($requestData['topic']);
$scheduledAt = trim($requestData['scheduled_at']);

// Normalize scheduled_at to ISO-8601 UTC
try {
    $dt = new DateTime($scheduledAt);
    $dt->setTimezone(new DateTimeZone('UTC'));
    $scheduledAtIso = $dt->format(DateTime::ATOM);
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'scheduled_at must be a valid date-time.'
    ]);
    exit();
}

// Prepare Firestore document payload
$nowUtc = (new DateTime('now', new DateTimeZone('UTC')))->format(DateTime::ATOM);

$firestoreBody = [
    'fields' => [
        'title' => ['stringValue' => $title],
        'message' => ['stringValue' => $message],
        'topic' => ['stringValue' => $topic],
        'scheduled_at' => ['timestampValue' => $scheduledAtIso],
        'status' => ['stringValue' => 'pending'],
        'created_at' => ['timestampValue' => $nowUtc],
    ],
];

try {
    $accessToken = getFirestoreAccessToken($serviceAccountFile);

    $url = sprintf(
        'https://firestore.googleapis.com/v1/projects/%s/databases/(default)/documents/scheduled_notifications',
        $projectId
    );

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $accessToken,
        'Content-Type: application/json; charset=utf-8',
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($firestoreBody));

    $result = curl_exec($ch);

    if ($result === false) {
        $error = curl_error($ch);
        curl_close($ch);
        throw new Exception('cURL error: ' . $error);
    }

    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $decoded = json_decode($result, true);

    if ($httpCode >= 200 && $httpCode < 300 && isset($decoded['name'])) {
        http_response_code(201);
        echo json_encode([
            'success' => true,
            'message' => 'Scheduled notification created',
            'document_name' => $decoded['name'],
            'firestore_response' => $decoded,
        ]);
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Failed to create scheduled notification in Firestore',
            'firestore_response' => $decoded,
        ]);
    }
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Server error: ' . $e->getMessage(),
    ]);
}


